// ==========================================================================
//
// = LIBRARY
//     PCIEFPGA
// 
// = FILENAME
//     PCIEFPGATxMemories.cpp
// 
// = COPYRIGHT
//     (C) Copyright 2003 Agilent Technologies
//
// ==========================================================================
#include "PCIEFPGATxMemories.h"
#include "PCIEMemInfo.h"
#include "devpcieintdefs.h"
#include "AgtPCIEControl.h"


//***************************************************************************
CPCIEFpgaTxMemories::CPCIEFpgaTxMemories()
{
  for(UInt32 ii=0; ii < PCIE_MAXIMUM_NUMBER_OF_PROBE_BOARDS; ii++)
  {
    for(UInt8 i=0; i < (UInt8) TXMEMORIES_NOFHWCHANNELS; i++)
    {
      mBlockRepeatFirst[ii][i] = 0x0;
      mBlockRepeatSecond[ii][i] = 0x0;
      mBlockRepeatThird[ii][i] = 0x0;
      mBlockRepeatFourth[ii][i] = 0x0;
      mBlockRepeatFifth[ii][i] = 0x0;

      mReqBehRepeatFirst[ii][i] = 0x0;
      mReqBehRepeatSecond[ii][i] = 0x0;
      mReqBehRepeatThird[ii][i] = 0x0;
      mReqBehRepeatFourth[ii][i] = 0x0;
      mReqBehRepeatFifth[ii][i] = 0x0;
    }
    for(UInt8 i=0; i < (UInt8) TXMEMORIES_NOFQUEUES; i++)
    {
      mCompBehRepeatFirst[ii][i] = 0x0;
      mCompBehRepeatSecond[ii][i] = 0x0;
    }
  }

  myController=new CAgtPCIEControl(); // returns only pointer if instance exists
}

//***************************************************************************
CPCIEFpgaTxMemories::~CPCIEFpgaTxMemories()
{
  delete myController;
  myController = NULL; 
}

// -----------------------------------------------------------------------
// Block memories 
// -----------------------------------------------------------------------
bool
CPCIEFpgaTxMemories::checkBlockPropValid(const EPCIEBlock& prop)
{
  switch(prop)
  {
  case PCIE_BLOCK_CFG_RSVD111:
  case PCIE_BLOCK_MEM64_RSVD151:
  case PCIE_BLOCK_MEM32_RSVD111:
  case PCIE_BLOCK_IO_RSVD111:
  case PCIE_BLOCK_MSGAS_RSVD047:
  case PCIE_BLOCK_MSGAS_RSVD087:
  case PCIE_BLOCK_MSGAS_RSVD127:
  /*case PCIE_BLOCK_MSG_BYTES08_11:
  case PCIE_BLOCK_MSG_BYTES12_15:*/
  case PCIE_BLOCK_MSGASD_RSVD047:
  case PCIE_BLOCK_MSGASD_RSVD087:
  case PCIE_BLOCK_MSGASD_RSVD127:
  case PCIE_BLOCK_OBSOLETE_34:
    return 0;
    break;
  default:
    return 1;
  }
}

// -----------------------------------------------------------------------
void
CPCIEFpgaTxMemories::setBlockProp(const AgtPortHandleT& portHandle, const EPCIEBlock& prop,
                                  const EPCIEHwChannelFunction& channel, const UInt32& line, const UInt32& val)
{
#ifdef LINUXOFFLINE_DEBUG   
  fprintf(stderr,"PCIEFPGATxMemories.cpp CPCIEFpgaTxMemories::setBlockProp\n");
#endif

  UInt32 theVal = val;
  UInt32 theLine = line;
  bool valid = 0;


  if(checkBlockPropValid(prop))
  {
    switch(prop)
    {
      case PCIE_BLOCK_MEM64_ADDRLO:
      case PCIE_BLOCK_MEM32_ADDR:
      case PCIE_BLOCK_IO_ADDR:
        theVal >>=2; // value is in bytes but must be written in DWORDs      
        valid = 1;
        break;
      case PCIE_BLOCK_INTADDR:
        {
          theVal >>=2; // value is in bytes but must be written in DWORDs
          UInt8 dw3 = 3;
          UInt8 dw4 = 4;
          UInt32 DW3Val = getBlockLineDW(portHandle,channel,theLine,dw3);  
          UInt32 DW4Val = getBlockLineDW(portHandle,channel,theLine,dw4);    
          CPCIEMemInfo::intAddrWrite(theVal,DW3Val,DW4Val);
          setBlockLineDW(portHandle,channel,theLine,dw3,DW3Val); 
          setBlockLineDW(portHandle,channel,theLine,dw4,DW4Val); 
        }
        break;
      case PCIE_BLOCK_RESOURCE:
        {
          switch((EPCIEBlockValResource)val)
          {
            case PCIE_BLOCKV_RESOURCE_GENERATOR:
              theVal = 0x2;
              break;
            case PCIE_BLOCKV_RESOURCE_MEM:
            case PCIE_BLOCKV_RESOURCE_DISCARD: // not yet implemented in HW !!
            default:
              theVal = 0x0; // bit 1=0 => datamem
          }
        } 
        valid = 1;
        break;
      case PCIE_BLOCK_REPEAT:
        {
          if(theLine > 0)
          {
             theLine--;
             valid = 1;
          }
          else if( ( PCIE_HWCHANNEL_FUNCTION_A == channel ) ||
                   ( PCIE_HWCHANNEL_FUNCTION_B == channel ) )
          {
            mBlockRepeatFirst[portHandle][(UInt8)channel] = val;
          }
          else if( ( PCIE_HWCHANNEL_FUNCTION_C == channel ) ||
                   ( PCIE_HWCHANNEL_FUNCTION_BVF1 == channel ) )
          {
            mBlockRepeatSecond[portHandle][(UInt8)channel] = val;
          }
          else if( ( PCIE_HWCHANNEL_FUNCTION_BVF2 == channel ) ||
                   ( PCIE_HWCHANNEL_FUNCTION_CVF1 == channel ) ||
                   ( PCIE_HWCHANNEL_FUNCTION_CVF2 == channel ) )
          {
            mBlockRepeatThird[portHandle][(UInt8)channel] = val;
          }
          else if( ( PCIE_HWCHANNEL_FUNCTION_D == channel ) ||
                   ( PCIE_HWCHANNEL_FUNCTION_E == channel ) ||
                   ( PCIE_HWCHANNEL_FUNCTION_DVF1 == channel ) ||
                   ( PCIE_HWCHANNEL_FUNCTION_DVF2 == channel ) )
          {
            mBlockRepeatFourth[portHandle][(UInt8)channel] = val;
          }
          else if( ( PCIE_HWCHANNEL_FUNCTION_EVF1 == channel ) ||
                   ( PCIE_HWCHANNEL_FUNCTION_EVF2 == channel ) )
          {
            mBlockRepeatFifth[portHandle][(UInt8)channel] = val;
          }
        }
        break;
      case PCIE_BLOCK_MCODE:
        {
          UInt8 hIdx = CPCIEMemInfo::getDWNum(PCIE_BLOCK_1STDWBE);
          UInt32 hVal = getBlockLineDW(portHandle,channel,theLine,hIdx);
          CPCIEMemInfo::setPropVal(PCIE_BLOCK_1STDWBE,hVal,(theVal & 0xf));
          CPCIEMemInfo::setPropVal(PCIE_BLOCK_LASTDWBE,hVal,( (theVal>>4) & 0xf));
          setBlockLineDW(portHandle,channel,theLine,hIdx,hVal);
        }
        break;
      default:
        valid = 1;
    }
  }
  
  if(valid)
  {
    // Get zero-based index of blockline DW
    UInt8 hIdx = CPCIEMemInfo::getDWNum(prop);

    // Get current blockline DW from HW
    UInt32 hVal = getBlockLineDW(portHandle,channel,theLine,hIdx);
    CPCIEMemInfo::setPropVal(prop,hVal,theVal); // modifies hVal only
    setBlockLineDW(portHandle,channel,theLine,hIdx,hVal);  // set modified blockline DW in HW 
  }
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaTxMemories::getBlockProp(const AgtPortHandleT& portHandle, const EPCIEBlock& prop, const EPCIEHwChannelFunction& channel, const UInt32& line)
{
  UInt32 theLine = line;
  
  if(prop == PCIE_BLOCK_INTADDR)
  {
    UInt8 dw3 = 3;
    UInt8 dw4 = 4;
    UInt32 DW3Val = getBlockLineDW(portHandle,channel,theLine,dw3);  
    UInt32 DW4Val = getBlockLineDW(portHandle,channel,theLine,dw4);    
    UInt32 tmp = CPCIEMemInfo::intAddrRead(DW3Val,DW4Val);
    tmp <<=2; //value is in DWORDs but return is expected in bytes
    return tmp;
  }
  
  if(prop == PCIE_BLOCK_REPEAT)
  {
    if(theLine > 0)
    {
      theLine--;
    }
    else if( ( PCIE_HWCHANNEL_FUNCTION_A == channel ) ||
             ( PCIE_HWCHANNEL_FUNCTION_B == channel ) )
    {
      return mBlockRepeatFirst[portHandle][(UInt8)channel];
    }
    else if( ( PCIE_HWCHANNEL_FUNCTION_C == channel ) ||
             ( PCIE_HWCHANNEL_FUNCTION_BVF1 == channel ) )
    { 
      return mBlockRepeatSecond[portHandle][(UInt8)channel];
    }
    else if( ( PCIE_HWCHANNEL_FUNCTION_BVF2 == channel ) ||
             ( PCIE_HWCHANNEL_FUNCTION_CVF1 == channel ) ||
             ( PCIE_HWCHANNEL_FUNCTION_CVF2 == channel ) )
    {
      return mBlockRepeatThird[portHandle][(UInt8)channel];
    }
    else if( ( PCIE_HWCHANNEL_FUNCTION_D == channel ) ||
             ( PCIE_HWCHANNEL_FUNCTION_E == channel ) ||
             ( PCIE_HWCHANNEL_FUNCTION_DVF1 == channel ) ||
             ( PCIE_HWCHANNEL_FUNCTION_DVF2 == channel ) )
    {
      return mBlockRepeatFourth[portHandle][(UInt8)channel];
    }
    else if( ( PCIE_HWCHANNEL_FUNCTION_EVF1 == channel ) ||
             ( PCIE_HWCHANNEL_FUNCTION_EVF2 == channel ) )
    {
      return mBlockRepeatFifth[portHandle][(UInt8)channel];
    }
  }

  if(prop == PCIE_BLOCK_MCODE)
  {
    UInt8 hIdx = CPCIEMemInfo::getDWNum(PCIE_BLOCK_1STDWBE);
    UInt32 hVal = getBlockLineDW(portHandle,channel,theLine,hIdx);
    UInt32 tmp = CPCIEMemInfo::getPropVal(PCIE_BLOCK_1STDWBE,hVal);
    tmp |= (CPCIEMemInfo::getPropVal(PCIE_BLOCK_LASTDWBE,hVal) << 4);
    return tmp;
  }

  if(checkBlockPropValid(prop))
  {
    UInt8 hIdx = CPCIEMemInfo::getDWNum(prop);
    UInt32 hVal = getBlockLineDW(portHandle,channel,theLine,hIdx); 
    UInt32 retVal = CPCIEMemInfo::getPropVal(prop,hVal);

    switch(prop)
    {
    case PCIE_BLOCK_MEM64_ADDRLO:
    case PCIE_BLOCK_MEM32_ADDR:
    case PCIE_BLOCK_IO_ADDR:
      retVal <<=2; //value is in DWORDs but return is expected in bytes 
      break;
    case PCIE_BLOCK_RESOURCE:
      {
        switch(retVal)
        {
          case 0x0:
            return (UInt32)PCIE_BLOCKV_RESOURCE_MEM;
          case 0x2:
            return (UInt32)PCIE_BLOCKV_RESOURCE_GENERATOR;
          default:
            return 0x0;
        }
      } 
      break;
    default:
      return retVal;
    }
    
    return retVal;
  }
  else
  {
    return 0x0;
  }
}

// -----------------------------------------------------------------------
void
CPCIEFpgaTxMemories::setBlockLineDefault(const AgtPortHandleT& portHandle, const EPCIEHwChannelFunction& channel, const UInt32& line)
{
  UInt32 theLine = line;
  UInt8 hIdx = 0;
  UInt32 hVal = 0;
  
  // prepare DW 0
  hIdx = 0; // we just KNOW this is correct (CPCIEMemInfo::getDWNum(prop);)
  hVal = getBlockLineDW(portHandle,channel,theLine,hIdx); // makes sure we don't overwrite repeat_next
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_RSVD007,hVal); // r31
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_FMT,hVal);    // fmt
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_TYPE,hVal);   // type
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_RSVD017,hVal);// r23
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_TC,hVal);     // tc
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_RSVD013,hVal);// r19_16
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_ATTR,hVal);   // attr
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_RSVD023,hVal);// r11_10
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_MARK,hVal);   // mark
  setBlockLineDW(portHandle,channel,theLine,hIdx,hVal);

  // prepare DW 1
  hIdx = 1; // we just KNOW this is correct (CPCIEMemInfo::getDWNum(prop);)
  hVal = 0;
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_MEM32_ADDR,hVal); // addr_lo
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_RESOURCE,hVal); // data_ind and data_comp
  setBlockLineDW(portHandle,channel,theLine,hIdx,hVal);
  
  // prepare DW 2
  hIdx = 2; // we just KNOW this is correct (CPCIEMemInfo::getDWNum(prop);)
  hVal = 0;
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_MEM64_ADDRHI,hVal); // addr_hi
  setBlockLineDW(portHandle,channel,theLine,hIdx,hVal);
  
  // prepare DW 3
  hIdx = 3; // we just KNOW this is correct (CPCIEMemInfo::getDWNum(prop);)
  hVal = 0;
  // int_addr(12:2) see below
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_LEN,hVal); // length
  setBlockLineDW(portHandle,channel,theLine,hIdx,hVal);

  // prepare DW 4
  hIdx = 4; // we just KNOW this is correct (CPCIEMemInfo::getDWNum(prop);)
  hVal = 0;
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_PATTERN_TERM,hVal); // patt
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_WAITREPEAT,hVal);   // wait/condstart/repeat
  CPCIEMemInfo::setIs64Bit(hVal,0); // addr_is_64
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_CFG_FUNCNUM,hVal); // fct_num
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_1STDWBE,hVal); // fdw_be
  CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_LASTDWBE,hVal); // ldw_be
  // int_addr(20:13) see below
  setBlockLineDW(portHandle,channel,theLine,hIdx,hVal);

  // prepares intaddr from DW3 and DW4
  UInt8 dw3 = 3;
  UInt8 dw4 = 4;
  UInt32 DW3Val = getBlockLineDW(portHandle,channel,theLine,dw3);  
  UInt32 DW4Val = getBlockLineDW(portHandle,channel,theLine,dw4);    
  CPCIEMemInfo::intAddrDefaultWrite(DW3Val,DW4Val);
  setBlockLineDW(portHandle,channel,theLine,dw3,DW3Val); 
  setBlockLineDW(portHandle,channel,theLine,dw4,DW4Val);

  
  // adapt repeat line for line-1
  if(line != 0)
  {
    theLine = line - 1;
    hIdx = 0;
    UInt32 hVal = getBlockLineDW(portHandle,channel,theLine,hIdx);
    CPCIEMemInfo::setPropDefaultVal(PCIE_BLOCK_REPEAT,hVal);
    setBlockLineDW(portHandle,channel,theLine,hIdx,hVal);
  }
  else
  {
    mBlockRepeatFirst[portHandle][(UInt8)channel]  = 0x0;
    mBlockRepeatSecond[portHandle][(UInt8)channel] =  0x0;
    mBlockRepeatThird[portHandle][(UInt8)channel]  =  0x0;
    mBlockRepeatFourth[portHandle][(UInt8)channel] =  0x0;
    mBlockRepeatFifth[portHandle][(UInt8)channel]  =  0x0;
  }
}
// -----------------------------------------------------------------------
UInt32 
CPCIEFpgaTxMemories::getBlockLineDW( const AgtPortHandleT& portHandle, const EPCIEHwChannelFunction& channel,
                                     const UInt32& line, const UInt8& dw )
{
  UInt32 val = 0;
  UInt32 offsetBytes = 4 * ( ( line * TXMEMORIES_BLOCKMEM_LINEWIDTH_DW ) + dw );
  
  switch( channel )
  {
  case PCIE_HWCHANNEL_FUNCTION_A:
    myController->FPGARead( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMA_OFFSET, val );
    break;
  case PCIE_HWCHANNEL_FUNCTION_B:
    myController->FPGARead( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMB_OFFSET, val );       
    break;
  case PCIE_HWCHANNEL_FUNCTION_C:
    myController->FPGARead( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMC_OFFSET, val );       
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF1:
    myController->FPGARead( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMBVF1_OFFSET, val );       
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF2:
    myController->FPGARead( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMBVF2_OFFSET, val );       
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF1:
    myController->FPGARead( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMCVF1_OFFSET, val );       
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF2:
    myController->FPGARead( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMCVF2_OFFSET, val );       
    break;
  case PCIE_HWCHANNEL_FUNCTION_D:
    myController->FPGARead( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMD_OFFSET, val );       
    break;
  case PCIE_HWCHANNEL_FUNCTION_E:
    myController->FPGARead( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEME_OFFSET, val );       
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF1:
    myController->FPGARead( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMDVF1_OFFSET, val );       
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF2:
    myController->FPGARead( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMDVF2_OFFSET, val );       
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF1:
    myController->FPGARead( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMEVF1_OFFSET, val );       
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF2:
    myController->FPGARead( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMEVF2_OFFSET, val );       
    break;
  default:
    AGT_THROW("Invalid getBlockLineDW index ");
  }
  return val;
}
  
// -----------------------------------------------------------------------
void
CPCIEFpgaTxMemories::setBlockLineDW( const AgtPortHandleT& portHandle, const EPCIEHwChannelFunction& channel,
                                     const UInt32& line, const UInt8& dw, const UInt32& val )
{
  UInt32 offsetBytes = 4 * ( ( line * TXMEMORIES_BLOCKMEM_LINEWIDTH_DW ) + dw );

  switch( channel )
  {
  case PCIE_HWCHANNEL_FUNCTION_A:
    myController->FPGAWrite( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMA_OFFSET, val );
    break;
  case PCIE_HWCHANNEL_FUNCTION_B:
    myController->FPGAWrite( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMB_OFFSET, val );
    break;
  case PCIE_HWCHANNEL_FUNCTION_C:
    myController->FPGAWrite( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMC_OFFSET, val );
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF1:
    myController->FPGAWrite( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMBVF1_OFFSET, val );
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF2:
    myController->FPGAWrite( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMBVF2_OFFSET, val );
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF1:
    myController->FPGAWrite( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMCVF1_OFFSET, val );
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF2:
    myController->FPGAWrite( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMCVF2_OFFSET, val );
    break;
  case PCIE_HWCHANNEL_FUNCTION_D:
    myController->FPGAWrite( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMD_OFFSET, val );
    break;
  case PCIE_HWCHANNEL_FUNCTION_E:
    myController->FPGAWrite( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEME_OFFSET, val );
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF1:
    myController->FPGAWrite( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMDVF1_OFFSET, val );
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF2:
    myController->FPGAWrite( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMDVF2_OFFSET, val );
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF1:
    myController->FPGAWrite( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMEVF1_OFFSET, val );
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF2:
    myController->FPGAWrite( portHandle, offsetBytes + PCIE_FPGATXBLOCKMEMEVF2_OFFSET, val );
    break;
  default:
    AGT_THROW("Invalid setBlockLineDW index ");
  }
}

// -----------------------------------------------------------------------
bool
CPCIEFpgaTxMemories::moveBlockLines(const AgtPortHandleT& portHandle, const EPCIEHwChannelFunction& channel,
                                    const UInt32& startLine, UInt32 stopLine,const bool& insert)
{
  UInt32 tmp = 0;
  
  if(insert)
  {
    if(stopLine == PCIE_BLOCKLINE_NUM)
    {
      stopLine--; // last line is lost 
    }
    
    for(UInt32 i = startLine; i <= stopLine; i++)
    {
      for(UInt8 dwNum = 0; dwNum < TXMEMORIES_BLOCKMEM_LINEWIDTH_ACTUALLY_USED_DW; dwNum++)
      {
        tmp = getBlockLineDW(portHandle,channel,stopLine - i,dwNum);
        setBlockLineDW(portHandle,channel,stopLine - i + 1,dwNum,tmp);
      }
    } 
    
    tmp = getBlockProp(portHandle,PCIE_BLOCK_REPEAT,channel,startLine);
    setBlockProp(portHandle,PCIE_BLOCK_REPEAT,channel,startLine+1,tmp);
    
  }
  else
  {
    UInt32 rememberRepeat = 0;
    
    rememberRepeat = getBlockProp(portHandle,PCIE_BLOCK_REPEAT,channel,startLine+1);
    
    for(UInt32 i = startLine+1; i <= stopLine; i++)
    {
      for(UInt8 dwNum = 0; dwNum < TXMEMORIES_BLOCKMEM_LINEWIDTH_ACTUALLY_USED_DW; dwNum ++)
      {
        tmp = getBlockLineDW(portHandle,channel,i,dwNum);
        setBlockLineDW(portHandle,channel,i-1,dwNum,tmp);
      }
    }
    
    setBlockProp(portHandle,PCIE_BLOCK_REPEAT,channel,startLine,rememberRepeat);
  }
  
  return 1; // success
}

// -----------------------------------------------------------------------
bool
CPCIEFpgaTxMemories::moveReqBehLines(const AgtPortHandleT& portHandle, const EPCIEHwChannelFunction& channel, const UInt32& startLine, UInt32 stopLine,const bool& insert)
{
  UInt32 tmp = 0;
  
  if(insert)
  {
    if(stopLine == PCIE_REQBEHLINE_NUM)
    {
      stopLine--; // last line is lost 
    }
    
    for(UInt32 i = startLine; i <= stopLine; i++)
    {
      for(UInt8 dwNum = 0; dwNum < TXMEMORIES_REQBEHMEM_LINEWIDTH_DW; dwNum++)
      {
        tmp = getReqBehLineDW(portHandle,channel,stopLine - i,dwNum);
        setReqBehLineDW(portHandle,channel,stopLine - i + 1,dwNum,tmp);
      }
    } 
    
    tmp = getReqBehProp(portHandle,PCIE_REQBEH_REPEAT,channel,startLine);
    setReqBehProp(portHandle,PCIE_REQBEH_REPEAT,channel,startLine+1,tmp);
  }
  else
  {
    UInt32 rememberRepeat = 0;
    
    rememberRepeat = getReqBehProp(portHandle,PCIE_REQBEH_REPEAT,channel,startLine+1);
    
    for(UInt32 i = startLine+1; i <= stopLine; i++)
    {
      for(UInt8 dwNum = 0; dwNum < TXMEMORIES_REQBEHMEM_LINEWIDTH_DW; dwNum ++)
      {
        tmp = getReqBehLineDW(portHandle,channel,i,dwNum);
        setReqBehLineDW(portHandle,channel,i-1,dwNum,tmp);
      }
    }
    
    setReqBehProp(portHandle,PCIE_REQBEH_REPEAT,channel,startLine,rememberRepeat);
  }
  
  return 1; // success
}

// -----------------------------------------------------------------------
bool
CPCIEFpgaTxMemories::moveCompBehLines(const AgtPortHandleT& portHandle, const EPCIECompQueue& queue, const UInt32& startLine, UInt32 stopLine,const bool& insert)
{
  UInt32 tmp = 0;

  if(insert)
  {
    if(stopLine == PCIE_COMPBEHLINE_NUM)
    {
      stopLine--; // last line is lost 
    }
    
    for(UInt32 i = startLine; i <= stopLine; i++)
    {
      for(UInt8 dwNum = 0; dwNum < TXMEMORIES_COMPBEHMEM_LINEWIDTH_DW; dwNum++)
      {
        tmp = getCompBehLineDW(portHandle,queue,stopLine - i,dwNum);
        setCompBehLineDW(portHandle,queue,stopLine - i + 1,dwNum,tmp);
      }
    } 
    
    tmp = getCompBehProp(portHandle,PCIE_COMPBEH_REPEAT,queue,startLine);
    setCompBehProp(portHandle,PCIE_COMPBEH_REPEAT,queue,startLine+1,tmp);
  }
  else
  {
    UInt32 rememberRepeat = 0;
    
    rememberRepeat = getCompBehProp(portHandle,PCIE_COMPBEH_REPEAT,queue,startLine+1);
    
    for(UInt32 i = startLine+1; i <= stopLine; i++)
    {
      for(UInt8 dwNum = 0; dwNum < TXMEMORIES_COMPBEHMEM_LINEWIDTH_DW; dwNum ++)
      {
        tmp = getCompBehLineDW(portHandle,queue,i,dwNum);
        setCompBehLineDW(portHandle,queue,i-1,dwNum,tmp);
      }
    }
    
    setCompBehProp(portHandle,PCIE_COMPBEH_REPEAT,queue,startLine,rememberRepeat);
  }
  
  return 1; // success
}

// -----------------------------------------------------------------------
bool
CPCIEFpgaTxMemories::insertBlockLine(const AgtPortHandleT& portHandle, const EPCIEHwChannelFunction& channel,const UInt32& linePos, const UInt32& validLineCount)
{
  if(validLineCount == 0 || linePos == validLineCount)
  {
    return 1; // success
  }

  if(linePos < validLineCount)
  {
    UInt32 startLine = linePos;
    UInt32 stopLine = validLineCount-1;
    return moveBlockLines(portHandle,channel,startLine,stopLine,1);
  }

  return 0;// no success
}

// -----------------------------------------------------------------------
bool
CPCIEFpgaTxMemories::removeBlockLine(const AgtPortHandleT& portHandle, const EPCIEHwChannelFunction& channel,const UInt32& linePos, const UInt32& validLineCount)
{
  if(validLineCount == 0 || linePos == validLineCount)
  {
    return 1; // success
  }
  
  if(linePos < validLineCount)
  {
    UInt32 startLine = linePos;
    UInt32 stopLine = validLineCount-1;
    moveBlockLines(portHandle,channel,startLine,stopLine,0);
    return 1;// success
  }

  return 0;// no success
}

// -----------------------------------------------------------------------
bool
CPCIEFpgaTxMemories::insertReqBehLine(const AgtPortHandleT& portHandle, const EPCIEHwChannelFunction& channel,const UInt32& linePos, const UInt32& validLineCount)
{
  if(validLineCount == 0 || linePos == validLineCount)
  {
    return 1; // success
  }

  if(linePos < validLineCount)
  {
    UInt32 startLine = linePos;
    UInt32 stopLine = validLineCount-1;
    return moveReqBehLines(portHandle,channel,startLine,stopLine,1);
  }

  return 0;// no success
}


// -----------------------------------------------------------------------
bool
CPCIEFpgaTxMemories::removeReqBehLine(const AgtPortHandleT& portHandle, const EPCIEHwChannelFunction& channel,const UInt32& linePos, const UInt32& validLineCount)
{
  if(validLineCount == 0 || linePos == validLineCount)
  {
    return 1; // success
  }
  
  if(linePos < validLineCount)
  {
    UInt32 startLine = linePos;
    UInt32 stopLine = validLineCount-1;
    moveReqBehLines(portHandle,channel,startLine,stopLine,0);
    return 1;// success
  }

  return 0;// no success
}

// -----------------------------------------------------------------------
bool
CPCIEFpgaTxMemories::insertCompBehLine(const AgtPortHandleT& portHandle, const EPCIECompQueue& queue,const UInt32& linePos, const UInt32& validLineCount)
{
  if(validLineCount == 0 || linePos == validLineCount)
  {
    return 1; // success
  }

  if(linePos < validLineCount)
  {
    UInt32 startLine = linePos;
    UInt32 stopLine = validLineCount-1;
    return moveCompBehLines(portHandle,queue,startLine,stopLine,1);
  }

  return 0;// no success
}


// -----------------------------------------------------------------------
bool
CPCIEFpgaTxMemories::removeCompBehLine(const AgtPortHandleT& portHandle, const EPCIECompQueue& queue,const UInt32& linePos, const UInt32& validLineCount)
{
  if(validLineCount == 0 || linePos == validLineCount)
  {
    return 1; // success
  }
  
  if(linePos < validLineCount)
  {
    UInt32 startLine = linePos;
    UInt32 stopLine = validLineCount-1;
    moveCompBehLines(portHandle,queue,startLine,stopLine,0);
    return 1;// success
  }

  return 0;// no success
}

// -----------------------------------------------------------------------
// Requester behaviour memories
// -----------------------------------------------------------------------
void
CPCIEFpgaTxMemories::setReqBehProp(const AgtPortHandleT& portHandle, const EPCIEReqBeh& prop, const EPCIEHwChannelFunction& channel, const UInt32& line, const UInt32& val)
{
  UInt32 theVal = val;
  UInt32 theLine = line;
  bool valid = 0;

  switch(prop)
  {
    
  // the tag and autotag feature are not available for Gen2
  case PCIE_REQBEH_TAG:
  case PCIE_REQBEH_AUTOTAG:
    {
      valid =0;
    }
    break;
  case PCIE_REQBEH_GAP:
    {
      UInt8 dw0 = 0;
      UInt8 dw1 = 1;
      UInt32 DW0Val = getReqBehLineDW(portHandle,channel,theLine,dw0);  
      UInt32 DW1Val = getReqBehLineDW(portHandle,channel,theLine,dw1);    
      CPCIEMemInfo::gapWrite(val,DW0Val,DW1Val);
        setReqBehLineDW(portHandle,channel,theLine,dw0,DW0Val); 
        setReqBehLineDW(portHandle,channel,theLine,dw1,DW1Val); 
      } 
      break;
    case PCIE_REQBEH_REPEAT:
      {
        if(theLine > 0)
        {
           theLine--;
           valid = 1;
        }
        else if( ( PCIE_HWCHANNEL_FUNCTION_A == channel ) ||
                 ( PCIE_HWCHANNEL_FUNCTION_B == channel ) )
        { 
          mReqBehRepeatFirst[portHandle][(UInt8)channel] = val;
        }
        else if( ( PCIE_HWCHANNEL_FUNCTION_C == channel ) ||
                 ( PCIE_HWCHANNEL_FUNCTION_BVF1 == channel ) )
        {
          mReqBehRepeatSecond[portHandle][(UInt8)channel] = val;
        }
        else if( ( PCIE_HWCHANNEL_FUNCTION_BVF2 == channel ) ||
                 ( PCIE_HWCHANNEL_FUNCTION_CVF1 == channel ) ||
                 ( PCIE_HWCHANNEL_FUNCTION_CVF2 == channel ) )
        {
          mReqBehRepeatThird[portHandle][(UInt8)channel] = val;
        }
        else if( ( PCIE_HWCHANNEL_FUNCTION_D == channel ) ||
                 ( PCIE_HWCHANNEL_FUNCTION_E == channel ) ||
                 ( PCIE_HWCHANNEL_FUNCTION_DVF1 == channel ) ||
                 ( PCIE_HWCHANNEL_FUNCTION_DVF2 == channel ) )
        {
          mReqBehRepeatFourth[portHandle][(UInt8)channel] = val;
        }
        else if( ( PCIE_HWCHANNEL_FUNCTION_EVF1 == channel ) ||
                 ( PCIE_HWCHANNEL_FUNCTION_EVF2 == channel ) )
        {
          mReqBehRepeatFifth[portHandle][(UInt8)channel] = val;
        }
      }
      break;
    default:
      valid =1;
  }

  if(valid)
  {
    UInt8 hIdx = CPCIEMemInfo::getDWNum(prop);
    UInt32 hVal = getReqBehLineDW(portHandle,channel,theLine,hIdx);
    CPCIEMemInfo::setPropVal(prop,hVal,theVal);
    setReqBehLineDW(portHandle,channel,theLine,hIdx,hVal);
  }
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaTxMemories::getReqBehProp(const AgtPortHandleT& portHandle, const EPCIEReqBeh& prop, const EPCIEHwChannelFunction& channel, const UInt32& line)
{
  UInt32 theLine = line;
  
  // the tag and autotag feature are not available for Gen2
  // here the autotag is always on, therefore return '1' when asked
  if(prop == PCIE_REQBEH_AUTOTAG)
  {
    return 0x1;
  }

  if(prop == PCIE_REQBEH_TAG)
  {
    return 0x0;
  }

  if(prop == PCIE_REQBEH_REPEAT)
  {
    if(theLine > 0)
    {
      theLine--;
    }
    else if( ( PCIE_HWCHANNEL_FUNCTION_A == channel ) ||
             ( PCIE_HWCHANNEL_FUNCTION_B == channel ) )
    {
      return mReqBehRepeatFirst[portHandle][(UInt8)channel];
    }
    else if( ( PCIE_HWCHANNEL_FUNCTION_C == channel ) ||
             ( PCIE_HWCHANNEL_FUNCTION_BVF1 == channel ) )
    {
      return mReqBehRepeatSecond[portHandle][(UInt8)channel];
    }
    else if( ( PCIE_HWCHANNEL_FUNCTION_BVF2 == channel ) ||
             ( PCIE_HWCHANNEL_FUNCTION_CVF1 == channel ) ||
             ( PCIE_HWCHANNEL_FUNCTION_CVF2 == channel ) )
    {
      return mReqBehRepeatThird[portHandle][(UInt8)channel];
    }
    else if( ( PCIE_HWCHANNEL_FUNCTION_D == channel ) ||
             ( PCIE_HWCHANNEL_FUNCTION_E == channel ) ||
             ( PCIE_HWCHANNEL_FUNCTION_DVF1 == channel ) ||
             ( PCIE_HWCHANNEL_FUNCTION_DVF2 == channel ) )
    {
      return mReqBehRepeatFourth[portHandle][(UInt8)channel];
    }
    else if( ( PCIE_HWCHANNEL_FUNCTION_EVF1 == channel ) ||
             ( PCIE_HWCHANNEL_FUNCTION_EVF2 == channel ) )
    {
      return mReqBehRepeatFifth[portHandle][(UInt8)channel];
    }
  }
  
  if(prop == PCIE_REQBEH_GAP)
  {
    UInt8 dw0 = 0;
    UInt8 dw1 = 1;
    UInt32 DW0Val = getReqBehLineDW(portHandle,channel,theLine,dw0);  
    UInt32 DW1Val = getReqBehLineDW(portHandle,channel,theLine,dw1);     
    return CPCIEMemInfo::gapRead(DW0Val,DW1Val);
  }
  
  UInt8 hIdx = CPCIEMemInfo::getDWNum(prop);
  UInt32 hVal = getReqBehLineDW(portHandle,channel,theLine,hIdx); 
  UInt32 retVal = CPCIEMemInfo::getPropVal(prop,hVal);


  return retVal; 
}

// -----------------------------------------------------------------------
void
CPCIEFpgaTxMemories::setReqBehLineDefault(const AgtPortHandleT& portHandle, const EPCIEHwChannelFunction& channel, const UInt32& line)
{
  UInt32 theLine = line;
  bool valid = 1;

  for(UInt8 i=0;i<=(UInt8)PCIE_REQBEH_LEN /* Last entry */ ;i++)
  {
    theLine = line;
    valid = 1;
    
    EPCIEReqBeh prop = (EPCIEReqBeh)i;
    if(prop == PCIE_REQBEH_REPEAT)
    {
      if(line == 0)
      {
        mReqBehRepeatFirst[portHandle][(UInt8)channel] = 0x0;
        mReqBehRepeatSecond[portHandle][(UInt8)channel] = 0x0;
        mReqBehRepeatThird[portHandle][(UInt8)channel]  = 0x0; 
        mReqBehRepeatFourth[portHandle][(UInt8)channel]  = 0x0; 
        mReqBehRepeatFifth[portHandle][(UInt8)channel]  = 0x0; 
        valid = 0;
      }
      else
      {
        theLine = line - 1;
      }
    }
    
    if(prop == PCIE_REQBEH_GAP)
    {
      UInt8 dw0 = 0;
      UInt8 dw1 = 1;
      UInt32 DW0Val = getReqBehLineDW(portHandle,channel,theLine,dw0);  
      UInt32 DW1Val = getReqBehLineDW(portHandle,channel,theLine,dw1);    
      CPCIEMemInfo::intAddrDefaultWrite(DW0Val,DW1Val);
      setReqBehLineDW(portHandle,channel,theLine,dw0,DW0Val); 
      setReqBehLineDW(portHandle,channel,theLine,dw1,DW1Val);
      valid = 0;
    }
    
    // the default for tag and autotag are zero so there is no problem for Gen2 
    // where this feature is not supported
    if(valid)
    {
      UInt8 hIdx = CPCIEMemInfo::getDWNum(prop);
      UInt32 hVal = getReqBehLineDW(portHandle,channel,theLine,hIdx);
      CPCIEMemInfo::setPropDefaultVal(prop,hVal);
      setReqBehLineDW(portHandle,channel,theLine,hIdx,hVal);
    }
  }
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaTxMemories::getReqBehLineDW(const AgtPortHandleT& portHandle, const EPCIEHwChannelFunction& channel,
                                     const UInt32& line, const UInt8& dw)
{
  UInt32 val = 0;
  UInt32 offsetBytes = 4 * ((line*TXMEMORIES_REQBEHMEM_LINEWIDTH_DW)+dw);
  
  switch(channel)
  {
  case PCIE_HWCHANNEL_FUNCTION_A:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMA_OFFSET,val);
    break;
  case PCIE_HWCHANNEL_FUNCTION_B:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMB_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_C:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMC_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF1:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMBVF1_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF2:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMBVF2_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF1:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMCVF1_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF2:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMCVF2_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_D:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMD_OFFSET,val);
    break;
  case PCIE_HWCHANNEL_FUNCTION_E:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEME_OFFSET,val);
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF1:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMDVF1_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF2:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMDVF2_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF1:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMEVF1_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF2:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMEVF2_OFFSET,val);       
    break;
  default:
    AGT_THROW("Invalid getReqBehLineDW index ");
  }
  return val;
}
  
// -----------------------------------------------------------------------
void
CPCIEFpgaTxMemories::setReqBehLineDW(const AgtPortHandleT& portHandle, const EPCIEHwChannelFunction& channel, const UInt32& line, const UInt8& dw, const UInt32& val)
{

  UInt32 offsetBytes = 4 * ((line*TXMEMORIES_REQBEHMEM_LINEWIDTH_DW)+dw);

  switch(channel)
  {
  case PCIE_HWCHANNEL_FUNCTION_A:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMA_OFFSET,val);    
    break;
  case PCIE_HWCHANNEL_FUNCTION_B:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMB_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_C:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMC_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF1:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMBVF1_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF2:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMBVF2_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF1:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMCVF1_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF2:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMCVF2_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_D:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMD_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_E:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEME_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF1:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMDVF1_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF2:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMDVF2_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF1:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMEVF1_OFFSET,val);       
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF2:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXREQBEHMEMEVF2_OFFSET,val);       
    break;
  default:
    AGT_THROW("Invalid setReqBehDW index ");
  }
}

// -----------------------------------------------------------------------
// Comp Behav memories
// -----------------------------------------------------------------------
bool
CPCIEFpgaTxMemories::checkCompBehPropValid(const EPCIECompBeh& prop)
{
  switch(prop)
  {
   //case PCIE_COMPBEH_OBSOLETE_05:
   case PCIE_COMPBEH_OBSOLETE_11:
   case PCIE_COMPBEH_OBSOLETE_12:
   case PCIE_COMPBEH_OBSOLETE_13:
    return 0;
  default:
    return 1;
  }
}

// -----------------------------------------------------------------------
void
CPCIEFpgaTxMemories::setCompBehProp(const AgtPortHandleT& portHandle, const EPCIECompBeh& prop, const EPCIECompQueue& queue, const UInt32& line, const UInt32& val)
{
  UInt32 theLine = line;
  bool valid = 1;
  
  if(prop == PCIE_COMPBEH_REPEAT)
  {
    if(theLine > 0)
    {
      theLine--;
    }
    else if( ( PCIE_COMPQUEUE_0 == queue ) ||
             ( PCIE_COMPQUEUE_1 == queue ) )
    {
      mCompBehRepeatFirst[portHandle][(UInt8)queue] = val;
      valid = 0;
    }
    else if( ( PCIE_COMPQUEUE_2 == queue ) ||
             ( PCIE_COMPQUEUE_3 == queue ) ||
             ( PCIE_COMPQUEUE_4 == queue ) )
    {
      mCompBehRepeatSecond[portHandle][(UInt8)queue] = val;
      valid = 0;
    }
  }
  
  if(checkCompBehPropValid(prop) && valid)
  {
    UInt32 theVal = val;
    UInt8 hIdx = CPCIEMemInfo::getDWNum(prop);
    UInt32 hVal = getCompBehLineDW(portHandle,queue,theLine,hIdx);
    CPCIEMemInfo::setPropVal(prop,hVal,theVal); 
    setCompBehLineDW(portHandle,queue,theLine,hIdx,hVal);
  }
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaTxMemories::getCompBehProp(const AgtPortHandleT& portHandle, const EPCIECompBeh& prop, const EPCIECompQueue& queue, const UInt32& line)
{
  UInt32 retVal = 0;
  UInt32 theLine = line;
  
  if(prop == PCIE_COMPBEH_REPEAT)
  {
    if(theLine > 0)
    {
      theLine--;
    }
    else if( ( PCIE_COMPQUEUE_0 == queue ) ||
             ( PCIE_COMPQUEUE_1 == queue ) )
    {
      return mCompBehRepeatFirst[portHandle][(UInt8)queue];
    }
    else if( ( PCIE_COMPQUEUE_2 == queue ) ||
             ( PCIE_COMPQUEUE_3 == queue ) ||
             ( PCIE_COMPQUEUE_4 == queue ) )
    {
      return mCompBehRepeatSecond[portHandle][(UInt8)queue];
    }
  }
  
  if(checkCompBehPropValid(prop))
  {
    UInt8 hIdx = CPCIEMemInfo::getDWNum(prop);
    UInt32 hVal = getCompBehLineDW(portHandle,queue,theLine,hIdx); 
    retVal = CPCIEMemInfo::getPropVal(prop,hVal);
  }
  return retVal; 
}

// -----------------------------------------------------------------------
void
CPCIEFpgaTxMemories::setCompBehLineDefault(const AgtPortHandleT& portHandle, const EPCIECompQueue& queue, const UInt32& line)
{
  UInt32 theLine = line;
  bool valid = 1;

  for( UInt8 i = 0; i <= ( UInt8 )PCIE_COMPBEH_REPEAT /* Last entry */; i++ )
  {
    EPCIECompBeh prop = ( EPCIECompBeh )i;

    if( checkCompBehPropValid( prop ) )
    {
      theLine = line;
      valid = 1;
      
      if( prop == PCIE_COMPBEH_REPEAT )
      {
        if( line == 0 )
        {
          mCompBehRepeatFirst[portHandle][( UInt8 )queue] = 0x0;
          mCompBehRepeatSecond[portHandle][( UInt8 )queue] = 0x0;
          valid = 0;
        }
        else
        {
          theLine = line - 1;
        }
      }
 
      if( valid )
      {
        UInt8 hIdx = CPCIEMemInfo::getDWNum( prop );
        UInt32 hVal = getCompBehLineDW( portHandle, queue, theLine, hIdx );
        CPCIEMemInfo::setPropDefaultVal( prop, hVal );
        setCompBehLineDW( portHandle, queue, theLine, hIdx, hVal );
      }
    }
  }
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaTxMemories::getCompBehLineDW(const AgtPortHandleT& portHandle, const EPCIECompQueue& queue, const UInt32& line, const UInt8& dw)
{
  UInt32 val = 0; 
  UInt32 offsetBytes = 4 * ((line*TXMEMORIES_COMPBEHMEM_LINEWIDTH_DW)+dw);

  switch(queue)
  {
  case PCIE_COMPQUEUE_0:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXCOMPBEHMEM_QUEUE0_OFFSET,val);
    break;
  case PCIE_COMPQUEUE_1:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXCOMPBEHMEM_QUEUE1_OFFSET,val);       
    break;
  case PCIE_COMPQUEUE_2:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXCOMPBEHMEM_QUEUE2_OFFSET,val);
    break;
  case PCIE_COMPQUEUE_3:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXCOMPBEHMEM_QUEUE3_OFFSET,val);
    break;
  case PCIE_COMPQUEUE_4:
    myController->FPGARead(portHandle,offsetBytes+PCIE_FPGATXCOMPBEHMEM_QUEUE4_OFFSET,val);
    break;
  default:
    AGT_THROW("Invalid getCompBehLineDW index ");
  }
  return val;
}
  
// -----------------------------------------------------------------------
void
CPCIEFpgaTxMemories::setCompBehLineDW(const AgtPortHandleT& portHandle, const EPCIECompQueue& queue, const UInt32& line, const UInt8& dw, const UInt32& val)
{
  UInt32 offsetBytes = 4 * ((line*TXMEMORIES_COMPBEHMEM_LINEWIDTH_DW)+dw);

  switch(queue)
  {
  case PCIE_COMPQUEUE_0:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXCOMPBEHMEM_QUEUE0_OFFSET,val);    
    break;
  case PCIE_COMPQUEUE_1:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXCOMPBEHMEM_QUEUE1_OFFSET,val);       
    break;
  case PCIE_COMPQUEUE_2:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXCOMPBEHMEM_QUEUE2_OFFSET,val);    
    break;
  case PCIE_COMPQUEUE_3:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXCOMPBEHMEM_QUEUE3_OFFSET,val);       
    break;
  case PCIE_COMPQUEUE_4:
    myController->FPGAWrite(portHandle,offsetBytes+PCIE_FPGATXCOMPBEHMEM_QUEUE4_OFFSET,val);       
    break;
  default:
    AGT_THROW("Invalid setCompBehDW index ");
  }
}

// -----------------------------------------------------------------------
void
CPCIEFpgaTxMemories::update64BitSettings(const AgtPortHandleT& portHandle, const EPCIEHwChannelFunction& channel, const UInt32& lineNum)
{
  
  UInt32 type = getBlockProp(portHandle,PCIE_BLOCK_TYPE,channel,lineNum);

  UInt8 hIdx = 4;
  UInt32 hVal = getBlockLineDW(portHandle,channel,lineNum,hIdx);

  CPCIEMemInfo::setIs64Bit(hVal,0);

  if(type == 0x0 || type == 0x1) // MRd, MRdLk or MWr
  {
    if(getBlockProp(portHandle,PCIE_BLOCK_FMT,channel,lineNum) & 0x1)
    {
      CPCIEMemInfo::setIs64Bit(hVal,1);
    }
  }

  setBlockLineDW(portHandle,channel,lineNum,hIdx,hVal); 
}